"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.extras = exports.parseScalabilityMode = exports.workerBin = exports.observer = exports.version = void 0;
exports.setLogEventListeners = setLogEventListeners;
exports.createWorker = createWorker;
exports.getSupportedRtpCapabilities = getSupportedRtpCapabilities;
const Logger_1 = require("./Logger");
const enhancedEvents_1 = require("./enhancedEvents");
const Worker_1 = require("./Worker");
const supportedRtpCapabilities_1 = require("./supportedRtpCapabilities");
const scalabilityModesUtils_1 = require("./scalabilityModesUtils");
const utils = require("./utils");
/**
 * Expose mediasoup version.
 */
// eslint-disable-next-line @typescript-eslint/no-require-imports
exports.version = require('../../package.json').version;
const observer = new enhancedEvents_1.EnhancedEventEmitter();
exports.observer = observer;
/**
 * Full path of the mediasoup-worker binary.
 */
var Worker_2 = require("./Worker");
Object.defineProperty(exports, "workerBin", { enumerable: true, get: function () { return Worker_2.workerBin; } });
const logger = new Logger_1.Logger();
/**
 * Set event listeners for mediasoup generated logs. If called with no arguments
 * then no events will be emitted.
 *
 * @example
 * ```ts
 * mediasoup.setLogEventListeners({
 *   ondebug: undefined,
 *   onwarn: (namespace: string, log: string) => {
 *     MyEnterpriseLogger.warn(`${namespace} ${log}`);
 *   },
 *   onerror: (namespace: string, log: string, error?: Error) => {
 *     if (error) {
 *       MyEnterpriseLogger.error(`${namespace} ${log}: ${error}`);
 *     } else {
 *       MyEnterpriseLogger.error(`${namespace} ${log}`);
 *     }
 *   }
 * });
 * ```
 */
function setLogEventListeners(listeners) {
    logger.debug('setLogEventListeners()');
    let debugLogEmitter;
    let warnLogEmitter;
    let errorLogEmitter;
    if (listeners?.ondebug) {
        debugLogEmitter = new enhancedEvents_1.EnhancedEventEmitter();
        debugLogEmitter.on('debuglog', listeners.ondebug);
    }
    if (listeners?.onwarn) {
        warnLogEmitter = new enhancedEvents_1.EnhancedEventEmitter();
        warnLogEmitter.on('warnlog', listeners.onwarn);
    }
    if (listeners?.onerror) {
        errorLogEmitter = new enhancedEvents_1.EnhancedEventEmitter();
        errorLogEmitter.on('errorlog', listeners.onerror);
    }
    Logger_1.Logger.setEmitters(debugLogEmitter, warnLogEmitter, errorLogEmitter);
}
/**
 * Create a Worker.
 */
async function createWorker({ logLevel = 'error', logTags, rtcMinPort = 10000, rtcMaxPort = 59999, dtlsCertificateFile, dtlsPrivateKeyFile, libwebrtcFieldTrials, disableLiburing, appData, } = {}) {
    logger.debug('createWorker()');
    if (appData && typeof appData !== 'object') {
        throw new TypeError('if given, appData must be an object');
    }
    const worker = new Worker_1.WorkerImpl({
        logLevel,
        logTags,
        rtcMinPort,
        rtcMaxPort,
        dtlsCertificateFile,
        dtlsPrivateKeyFile,
        libwebrtcFieldTrials,
        disableLiburing,
        appData,
    });
    return new Promise((resolve, reject) => {
        worker.on('@success', () => {
            // Emit observer event.
            observer.safeEmit('newworker', worker);
            resolve(worker);
        });
        worker.on('@failure', reject);
    });
}
/**
 * Get a cloned copy of the mediasoup supported RTP capabilities.
 */
function getSupportedRtpCapabilities() {
    return utils.clone(supportedRtpCapabilities_1.supportedRtpCapabilities);
}
/**
 * Expose parseScalabilityMode() function.
 */
var scalabilityModesUtils_2 = require("./scalabilityModesUtils");
Object.defineProperty(exports, "parseScalabilityMode", { enumerable: true, get: function () { return scalabilityModesUtils_2.parseScalabilityMode; } });
/**
 * Expose extras module.
 */
exports.extras = require("./extras");
// NOTE: This constant of type Index is created just to check at TypeScript
// level that everything exported here (all but TS types) matches the Index
// interface exposed by indexTypes.ts.
//
// eslint-disable-next-line @typescript-eslint/no-unused-vars
const indexImpl = {
    version: exports.version,
    observer,
    workerBin: Worker_1.workerBin,
    setLogEventListeners,
    createWorker,
    getSupportedRtpCapabilities,
    parseScalabilityMode: scalabilityModesUtils_1.parseScalabilityMode,
};
